<?php

namespace App\Http\Controllers;


use App\Http\Controllers\Controller;
use App\Models\Governorate;
use App\Models\Role;
use App\Models\User;
use App\Models\UserWorkSession;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class DashboardController extends Controller
{

    const _WIDGET_DAYS_LIMIT = 6;

    public function dashboard()
    {
        $user = auth()->guard('admin')->user();
        $employees  = User::query();
        if($user->role_id == Role::SUPERVISOR_ROLE){
            $governorates = Governorate::query()->where('id', $user->gov_id);
            $employees = $employees->whereIn('gov_id', function ($qu) use($user) {
                $qu->select('id')->from((new Governorate())->getTable())->where('id', $user->gov_id);
            });
        }else{
            $governorates = Governorate::query();
        }
        $totalEmployees = $employees->clone();
        $totalEmployees = $totalEmployees->count();
        $activeEmployees = $employees->clone();
        $activeEmployees = $activeEmployees->where('is_active', 1)->count();
        $governorates = $governorates->whereIn('id', function($qu) {
            $qu->select('gov_id')->from((new User())->getTable())->where('is_active', 1);
        })->get();
        $widgetDateAttendanceData = $this->getWeekAttendanceSummary();
        $widgetMonthlyReportAttendanceData = [];
        foreach ($governorates as $governorate) {
            $widgetMonthlyReportAttendanceData[$governorate->id] = $this->getGovMonthAttendanceSummary($governorate->id);
        }
        $colors = [
            1 => ["class" => "success", "name" => "green", "code" => "#50cd89"],
            2 => ["class" => "primary", "name" => "blue", "code" => "#009ef7"],
            3 => ["class" => "info", "name" => "purple", "code" => "#7239EA"],
            4 => ["class" => "warning", "name" => "yellow", "code" => "#ffc700"],
            5 => ["class" => "danger", "name" => "red", "code" => "#F1416C"],
            6 => ["class" => "dark", "name" => "black", "code" => "#1E2129"],
        ];

        return view('dashboard.index', compact('governorates', 'totalEmployees', 'activeEmployees', 'widgetDateAttendanceData', 'widgetMonthlyReportAttendanceData', 'colors'));
    }

    private function getWeekAttendanceSummary()
    {
        $user = auth()->guard('admin')->user();
        $currentDate = \Carbon\Carbon::now();
        $agoDate = $currentDate->subDays($currentDate->dayOfWeek)->subDays(self::_WIDGET_DAYS_LIMIT)->format('Y-m-d');
        $data = UserWorkSession::query()->where('start_date', '>=', $agoDate);
        if($user->role_id == Role::SUPERVISOR_ROLE){
            $data = $data->whereIn('user_id', function ($qu) use($user) {
                $qu->select('id')->from((new User())->getTable())->where('gov_id', $user->gov_id)->where('is_active', 1);
            });
        }else{
            $data = $data->whereIn('user_id', function ($qu) use($user) {
                $qu->select('id')->from((new User())->getTable())->where('is_active', 1);
            });
        }
        $data = $data->get();
        $attendanceData = [];
        foreach ($data as $datum) {
            $attendanceData[$datum->start_date->format('Y-m-d')][$datum->user->gov_id][] = $datum;
        }
        return $attendanceData;
    }

    private function getMonthAttendanceSummary()
    {
        $user = auth()->guard('admin')->user();
        $currentDate = \Carbon\Carbon::now();
        $startDate = $currentDate->clone()->startOfMonth()->format('Y-m-d');
        $data = UserWorkSession::query()->whereDate('start_date', '>=', $startDate)->whereDate('start_date', '<=', $currentDate->format('Y-m-d'));
        if($user->role_id == Role::SUPERVISOR_ROLE){
            $data = $data->whereIn('user_id', function ($qu) use($user) {
                $qu->select('id')->from((new User())->getTable())->where('gov_id', $user->gov_id)->where('is_active', 1);
            });
        }else{
            $data = $data->whereIn('user_id', function ($qu) use($user) {
                $qu->select('id')->from((new User())->getTable())->where('is_active', 1);
            });
        }
        $data = $data->get();
        $attendanceData = [];
        foreach ($data as $datum) {
            if(!key_exists($datum->user->gov_id, $attendanceData) || (is_array($attendanceData[$datum->user->gov_id]) && !key_exists($datum->user_id, $attendanceData[$datum->user->gov_id])) ){
                $attendanceData[$datum->user->gov_id][$datum->user_id] = 1;
            }else{
                $attendanceData[$datum->user->gov_id][$datum->user_id]++;
            }
//            $attendanceData[$datum->user->gov_id][$datum->user_id][] = $datum;
        }
        return $attendanceData;
    }

    private function getGovMonthAttendanceSummary($govId)
    {
        $user = auth()->guard('admin')->user();
        $currentDate = \Carbon\Carbon::now();
        $startDate = $currentDate->clone()->startOfMonth()->format('Y-m-d');
        $attendanceData = UserWorkSession::query()->select(['user_id', DB::raw('count(*) as total')])
            ->whereIn('user_id', function($qu) use($govId){
                $qu->select('id')->from((new User())->getTable())->where('is_active', 1)->where('gov_id', $govId);
            })
            ->whereDate('start_date', '>=', $startDate)
            ->whereDate('start_date', '<=', $currentDate->format('Y-m-d'))
            ->groupBy('user_id')
            ->orderBy('total', 'desc')
            ->get();
        $data = [];
        if($attendanceData->count() > 0){
            $data['most_attendance'] = $attendanceData[0]->user_id;
            $data['less_attendance'] = $attendanceData[array_key_last($attendanceData->toArray())]->user_id;
        }else{
            $data['most_attendance'] = null;
            $data['less_attendance'] = null;
        }
        $attendanceData = UserWorkSession::query()->select(['user_id', DB::raw('count(*) as total')])
            ->whereIn('user_id', function($qu) use($govId){
                $qu->select('id')->from((new User())->getTable())->where('is_active', 1)->where('gov_id', $govId);
            })
            ->whereDate('start_date', '>=', $startDate)
            ->whereDate('start_date', '<=', $currentDate->format('Y-m-d'))
            ->where('start_online', 1)
            ->where('end_online', 1)
            ->groupBy('user_id')
            ->orderBy('total', 'desc')
            ->get();
        if($attendanceData->count() > 0){
            $data['most_online'] = $attendanceData[0]->user_id;
            $data['less_online'] = $attendanceData[array_key_last($attendanceData->toArray())]->user_id;
        }else{
            $data['most_online'] = null;
            $data['less_online'] = null;
        }
        $attendanceData = UserWorkSession::query()->select(['user_id', DB::raw('count(*) as total')])
            ->whereIn('user_id', function($qu) use($govId){
                $qu->select('id')->from((new User())->getTable())->where('is_active', 1)->where('gov_id', $govId);
            })
            ->whereDate('start_date', '>=', $startDate)
            ->whereDate('start_date', '<=', $currentDate->format('Y-m-d'))
            ->where(function ($query) {
                $query->where('start_online', '=', 0)
                    ->orWhere('end_online', '=', 1);
            })
            ->groupBy('user_id')
            ->orderBy('total', 'desc')
            ->get();
        if($attendanceData->count() > 0){
            $data['most_offline'] = $attendanceData[0]->user_id;
            $data['less_offline'] = $attendanceData[array_key_last($attendanceData->toArray())]->user_id;
        }else{
            $data['most_offline'] = null;
            $data['less_offline'] = null;
        }
        return $data;
    }

    public function profile()
    {
        return view('users.profile');
    }

    public function saveProfile(Request $request)
    {
        $user = Auth::guard('admin')->user();
        $result = [];
        if($request->password != $request->confirm_password){
            $result = ["error" => true, "message" => __('New password & Confirm is not matched')];
        }else{
            if (isset($request->current_password)) {
                if (Hash::check($request->current_password, $user->password)) {
                    if ($request->password != null && $request->password != '') {
                        $user->update([
                            "password" => Hash::make($request->password)
                        ]);
                        $result = ["success" => true, "message" => __("Password Changed Successfully!")];
                    }
                } elseif ($request->current_password != null) {
                    $result = ["error" => true, "message" => __('Old password is wrong')];
                }
            } else {
                $result = ["error" => true, "message" => __('New password is empty')];
            }
        }

        return response()->json($result);
    }
}
