<?php

namespace App\DataTables;

use App\Models\Governorate;
use App\Models\Role;
use App\Models\Task;
use App\Models\User;
use App\Models\UserWorkSession;
use Defuse\Crypto\Crypto;
use Defuse\Crypto\Key;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class WorkSessionDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('action',function ($data){
                return $this->getActionColumn($data);
            })
//            ->addColumn('created_at', function(User $user) {
//                return $user->created_at != null ? $user->created_at->format('Y-m-d'):"";
//            })
//            ->addColumn('role', function(User $user) {
//                $role = $user->role_id != null ? $user->role->defaultLang->role_name:"";
//                if($user->role_id == Role::ADMIN_ROLE)
//                    $html_code = '<span class="badge badge-light-danger me-auto">'.$role.'</span>';
//                elseif($user->role_id == Role::EMPLOYEE_ROLE)
//                    $html_code = '<span class="badge badge-light-success me-auto">'.$role.'</span>';
//                else
//                    $html_code = '<span class="badge badge-light-warning me-auto">'.$role.'</span>';
//
//                return view('components.datatable.html-code', compact('html_code'));
//            })
            ->editColumn('user.full_name', function(UserWorkSession $session){
                if($session->user_id != null){
                    $user = $session->user;
                    return User::getDecryptedData($user->full_name);
                }else{
                    return "N/A";
                }
            })
            ->addColumn('governorate', function(UserWorkSession $session){
                return $session->user_id != null ? $session->user->governorate->defaultLang->name:"";
            })
            ->addColumn('date', function(UserWorkSession $session){
                if($session->start_date != null){
                    return $session->start_date->format('Y-m-d');
                }else{
                    return "N/A";
                }
            })
            ->editColumn('start_time', function(UserWorkSession $session){
                if($session->start_date != null){
                    return $session->start_date->format('H:i:s');
                }else{
                    return "N/A";
                }
            })
            ->editColumn('end_time', function(UserWorkSession $session){
                if($session->end_date != null){
                    return $session->end_date->format('H:i:s');
                }else{
                    return "N/A";
                }
            })
            ->editColumn('start_online', function (UserWorkSession $session){
                if($session->start_online)
                    $html_code = '<span class="badge badge-light-success me-auto">'.__("Online").'</span>';
                else
                    $html_code = '<span class="badge badge-light-danger me-auto">'.__("Offline").'</span>';

                return view('components.datatable.html-code', compact('html_code'));
            })
            ->editColumn('end_online', function (UserWorkSession $session){
                if($session->end_date != null ){
                    if($session->end_online)
                        $html_code = '<span class="badge badge-light-success me-auto">'.__("Online").'</span>';
                    else
                        $html_code = '<span class="badge badge-light-danger me-auto">'.__("Offline").'</span>';
                }else{
                    $html_code = '<span class="me-auto">'.__("N/A").'</span>';
                }


                return view('components.datatable.html-code', compact('html_code'));
            })
            ->filter(function ($query, $keyword) {
                $keyword = User::getEncryptedData($keyword);
                $query->whereIn('user_id', function($q) use($keyword){
                    $q->select('id')->from((new User())->getTable())
                        ->whereRaw('user.full_name like ?', ["%$keyword%"]);
                });

            })
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     * @param UserWorkSession $model
     * @return QueryBuilder
     */
    public function query(UserWorkSession $model): QueryBuilder
    {
        $query = $model->newQuery()->with('user');
        $gov_id = $this->request()->get('gov_id');
        $start_date = $this->request()->get('start_date');
        $end_date = $this->request()->get('end_date');
        $employee = $this->request()->get('employee');
        $user = auth()->guard('admin')->user();
        if($user->role_id == Role::SUPERVISOR_ROLE){
            $query->whereIn('user_id', function ($qu) use($user){
                $qu->select('id')->from((new user())->getTable())
                    ->where('gov_id', $user->gov_id);
            });
        }
        if($this->user != null){
            $query = $query->where('user_id', $this->user->id);
        }
        if($gov_id != null){
            $query = $query->whereIn('user_id', function($q) use($gov_id){
                $q->select('id')->from((new User())->getTable())->where('gov_id', $gov_id);
            });
        }
        if($employee != null){
            $query = $query->where('user_id', $employee);
//            $employee = User::getEncryptedData($employee);
//            $query = $query->whereIn('user_id', function($q) use($employee){
//                $q->select('id')->from((new User())->getTable())
//                    ->whereRaw('user.full_name like ?', ["%$employee%"]);
//            });
        }
        if($start_date == $end_date && ($start_date != null)) {
            $query = $query->whereDate('start_date', $start_date);
        }else{
            if($start_date != null){
                $query = $query->whereDate('start_date', '>=', $start_date);
            }
            if($end_date != null){
                $query = $query->whereDate('start_date', '<=', $end_date);
            }
        }

        return $query;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
                    ->setTableId('user-session-table')
                    ->orderBy('0', 'desc')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    ->pageLength(50)
                    ->dom('Bfrtipl')
                    ->orderBy(0, 'asc')
//                    ->setTableHeadClass('fw-bold text-muted fs-4')
                    ->setTableAttribute(['class' => 'table table-row-dashed table-row-gray-300 align-middle gs-0 gy-4'])
                    ->lengthMenu([10, 25, 50, 100, 500, 1000])
                    ->buttons([
                        Button::make('excel'),
                        Button::make('csv'),
                        Button::make('pdf'),
                        Button::make('print')
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('id'),
            Column::make('full_name')->data('user.full_name')->searchable(false),
            Column::make('governorate'),
            Column::make('date'),
            Column::make('start_time')->searchable(false),
            Column::make('end_time')->searchable(false),
            Column::make('start_online'),
            Column::make('end_online'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->title('actions')
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'User_sessions_' . date('YmdHis');
    }

    /**
     * @param $data
     * @return string
     */
    protected function getActionColumn($data): string
    {
        $buttons = [];
        $buttons[] = ["type" => 'link', 'route' => "#map_session_location", 'icon' => 'geolocation', 'label' => 'check', 'class'=>'view-on-map', 'attributes' => ['data-start-location' => $data->start_location, 'data-end-location' => $data->end_location]];
        $buttons[] = ["type" => 'link', 'route' => route('users.view-session', ['session' => $data->id]), 'attributes' => ['target' => '_blank'], 'icon' => 'eye', 'label' => 'view'];
//        if($data->is_active == 1){
//            $buttons[]  = ["type" => 'ajax', 'route' => route('users.activate'), 'icon' => 'flag', 'label' => 'activate', 'id' => $data->id];
//        }elseif($data->result == 'pending'){
//            $buttons[]  = ["type" => 'ajax', 'route' => route('users.deactivate'), 'icon' => 'flag', 'label' => 'deactivate', 'id' => $data->id];
//        }
//        $buttons[] = ["type" => 'link', 'route' => route('users.edit', ['user' => $data->id]), 'icon' => 'pencil', 'label' => 'edit', 'id' => $data->id];
//        $buttons[] = ["type" => 'ajax', 'route' => route('users.delete', ['id' => $data->id]), 'icon' => 'trash', 'label' => 'delete', 'id' => $data->id];

        return view('components.datatable.action-buttons', compact('buttons'));
    }
}
